(function (tracing, ui) {

  var global = this;

  global.viewEl;
  global.webSocket;
  global.startBtn;
  global.reconnectBtn;

  String.prototype.startswith = function (prefix) { return this.substring(0, prefix.length) == prefix; };

  var $ = function (selector, ancestor) {
    if (!ancestor) {
      ancestor = document;
    }

    if (selector.startswith('#')) {
      return ancestor.getElementById(selector.substring(1));
    } else if (selector.startswith('.')) {
      return ancestor.getElementsByClassName(selector.substring(1));
    } else {
      return ancestor.getElementsByTagName(selector);
    }
  };

  var displayError = function (title, errmsg) {
    var overlay = new ui.Overlay();
    overlay.textContent = errmsg;
    overlay.title = title;
    overlay.visible = true;
  };

  var createViewFromTraces = function (traces) {
    var m = new tracing.TraceModel();
    var p = m.importTracesWithProgressDialog(traces, true);
    p.then(function () {
      global.viewEl.model = m;
      global.viewEl.tabIndex = 1;
      if (global.viewEl.timeline) {
        global.viewEl.timeline.focusElement = global.viewEl;
      }
    }, function(err) {
      displayError('Trace Parsing Error', normalizeException(err).message);
    });
  };

  var showReconnectBtn = function () { global.reconnectBtn.style.display = 'block'; };
  var hideReconnectBtn = function () { global.reconnectBtn.style.display = 'none'; };
  var enableStartBtn = function () { global.startBtn.removeAttribute('disabled'); };
  var disableStartBtn = function () { global.startBtn.setAttribute('disabled', 'disabled'); };

  var reset = function () {
    disableStartBtn();

    if (global.webSocket) {
      global.webSocket.close();
    }
  };

  var emptyView = function () {
    // empty the result
    createViewFromTraces([]);
  };

  global.traceBucket = [];
  var handleWebSocketMessage = function (msg) {
    global.traceBucket.push(JSON.parse(msg));
  };

  global.started = false;
  global.traceDialog = null;

  var stop = function (e) {
    if (global.webSocket && global.started) {
      global.webSocket.send("stop");
      enableStartBtn();

      // display traces
      createViewFromTraces([JSON.stringify(global.traceBucket)]);

      // close trace dialog
      global.traceDialog.visible = false;

      global.started = false;
    }
  };

  var start = function (e) {
    if (global.webSocket && !global.started) {
      global.started = true;

      global.traceBucket = [];
      emptyView();

      global.webSocket.send("start");
      disableStartBtn();

      var createTraceDialog = function () {
        var overlay = new ui.Overlay();
        overlay.userCanClose = false;
        overlay.textContent = "Tracing active...";
        overlay.visible = false;

        var stopBtn = document.createElement('button');
        stopBtn.textContent = 'Stop tracing';
        stopBtn.onclick = stop;
        overlay.rightButtons.appendChild(stopBtn);

        return overlay;
      };

      if (!global.traceDialog) {
        global.traceDialog = createTraceDialog();
      }
      global.traceDialog.visible = true;
    }
  };

  var connect = function (addr) {
    global.webSocket = new WebSocket(addr);

    global.webSocket.onopen = function () {
      enableStartBtn();

      hideReconnectBtn();
    };
    global.webSocket.onclose = function () {
      if (global.started) {
        stop();
      }

      reset();
      displayError('Connection Error', 'Connection closed or refused.');
      global.webSocket = null;

      showReconnectBtn();
    };
    global.webSocket.onerror = function () {
      console.log("WebSocket connection error occurred.");
    };
    global.webSocket.onmessage = function (e) {
      handleWebSocketMessage(e.data);
    };
  };

  var tryConnect = function () {
    var protocol = location.protocol.startswith('https') ? 'wss://' : 'ws://';
    var addr = protocol + location.host;
    connect(addr);
  };

  var createStartBtn = function () {
    var btn = document.createElement('button');
    btn.textContent = 'Start tracing';
    btn.className = 'start';
    btn.onclick = start;
    btn.setAttribute('disabled', 'disabled');
    return btn;
  };

  var createReconnectBtn = function () {
    var btn = document.createElement('button');
    btn.textContent = 'Reconnect';
    btn.className = 'reconnect';
    btn.onclick = tryConnect;
    return btn;
  };

  var initialize = function () {
    global.viewEl = document.querySelector('.view');
    ui.decorate(global.viewEl, tracing.TimelineView);

    global.startBtn = createStartBtn();
    global.reconnectBtn = createReconnectBtn();

    global.viewEl.leftControls.appendChild(global.reconnectBtn);
    global.viewEl.leftControls.appendChild(global.startBtn);

    emptyView();
    reset();

    tryConnect();
  };

  window.onload = initialize;
  window.onhashchange = reset;

}(tracing, ui));
