var ReadingToolsCore =


{


	isRootPage: false,


	isDebug: false,


	iframeLoads: 0,


	convertLinksToFootnotes: false,


	cacheDiv:  null,   


	flags: 0x1 | 0x2,  








	FLAG_STRIP_UNLIKELYS: 0x1,


	FLAG_WEIGHT_CLASSES:  0x2,


}; 





/**


* All of the regular expressions in use within ReadingToolsCore.


* Defined up here so we don't instantiate them repeatedly in loops.


**/


var regexps ={


	unlikelyCandidatesRe:   /combx|comment|disqus|foot|header|menu|rss|shoutbox|sidebar|sponsor/i,


	okMaybeItsACandidateRe: /and|article|body|column|main/i,


	positiveRe:			 /article|body|content|entry|hentry|page|pagination|post|text/i,


	negativeRe:			 /combx|comment|contact|foot|footer|footnote|link|masthead|media|meta|promo|related|scroll|shoutbox|sponsor|tags|widget/i,


	divToPElementsRe:	   /<(a|blockquote|dl|div|img|ol|p|pre|table|ul)/i,


	replaceBrsRe:		   /(<br[^>]*>[ \n\r\t]*){2,}/gi,


	replaceFontsRe:		 /<(\/?)font[^>]*>/gi,


	trimRe:				 /^\s+|\s+$/g,


	normalizeRe:			/\s{2,}/g,


	killBreaksRe:		   /(<br\s*\/?>(\s|&nbsp;?)*){1,}/g,


	videoRe:				/http:\/\/(www\.)?(youtube|vimeo)\.com/i,


	skipFootnoteLinkRe:	 /^\s*(\[?[a-z0-9]{1,2}\]?|^|edit|citation needed)\s*$/i


};





/**


* Runs ReadingToolsCore.


*


* Workflow:


*  1. Prep the document by removing script tags, css, etc.


*  2. Build ReadingToolsCore's DOM tree.


*  3. Grab the article content from the current dom tree.


*  4. Replace the current DOM tree with the new one.


*  5. Read peacefully.


*


* @return void


**/


ReadingToolsCore.init = function()


{


	if((window.location.protocol + "//" + window.location.host + "/") == window.location.href)


		ReadingToolsCore.isRootPage = true;





	if(document.getElementById("ID_RT_ReadInner"))


		return;





	if(ReadingToolsCore.isRootPage == false)


		ReadingToolsCore.prepDocument();





	var articleTitle = ReadingToolsCore.getArticleTitle();


	var articleContent = ReadingToolsCore.getArticleContent();





	if(ReadingToolsCore.isRootPage == false)


	{


		/* So we have all of the content that we need. Now we clean it up for presentation. */


		ReadingToolsCore.prepArticle(articleContent);





		/**


		* If we attempted to strip unlikely candidates on the first run through, and we ended up with no content,


		* that may mean we stripped out the actual content so we couldn't parse it. So re-run init while preserving


		* unlikely candidates to have a better shot at getting our content out properly.


		**/


		if(ReadingToolsCore.getInnerText(articleContent, false).length < 250)


		{


			if (ReadingToolsCore.flagIsActive(ReadingToolsCore.FLAG_STRIP_UNLIKELYS))


			{


				ReadingToolsCore.removeFlag(ReadingToolsCore.FLAG_STRIP_UNLIKELYS);


				ReadingToolsCore.cacheDiv.innerHTML = document.body.innerHTML;


				return ReadingToolsCore.init();


			}


			else if (ReadingToolsCore.flagIsActive(ReadingToolsCore.FLAG_WEIGHT_CLASSES))


			{


				ReadingToolsCore.removeFlag(ReadingToolsCore.FLAG_WEIGHT_CLASSES);


				ReadingToolsCore.cacheDiv.innerHTML = document.body.innerHTML;


				return ReadingToolsCore.init();


			}


			else


			{


				articleContent.innerHTML = "Not Found";


			}


		}





		if(typeof(readConvertLinksToFootnotes) !== 'undefined' && readConvertLinksToFootnotes == true)


			ReadingToolsCore.convertLinksToFootnotes = true;


	}





	var innerDiv = document.createElement("DIV");


	innerDiv.id = "ID_RT_ReadInner";


	innerDiv.style.display = "none";





	/* Glue the structure of our document together. */


	innerDiv.appendChild(articleTitle);


	innerDiv.appendChild(articleContent);





	/* Clear the old HTML, insert the new content. */


	document.body.appendChild(innerDiv);





//	if(ReadingToolsCore.convertLinksToFootnotes && !window.location.href.match(/wikipedia\.org/g))


		ReadingToolsCore.addFootnotes(articleContent);


}





/**


* Prepare the HTML document for ReadingToolsCore to scrape it.


* This includes things like stripping javascript, CSS, and handling terrible markup.


*


* @return void


**/


ReadingToolsCore.prepDocument = function()


{


	if(!ReadingToolsCore.cacheDiv)


	{


		ReadingToolsCore.cacheDiv = document.createElement("DIV");


		if(document.body)


			ReadingToolsCore.cacheDiv.innerHTML = document.body.innerHTML;


	}





	/* Convert to body element if frame element is found */


	var frames = document.getElementsByTagName('frame');


	if(frames.length > 0)


	{


		var bestFrame = null;


		var bestFrameSize = 0;


		var frameLength = frames.length;


		for(var frameIndex = 0; frameIndex < frameLength; frameIndex++)


		{


			var frameSize = frames[frameIndex].offsetWidth + frames[frameIndex].offsetHeight;


			var canAccessFrame = false;





			try


			{


				frames[frameIndex].contentWindow.document.body;


				canAccessFrame = true;


			}


			catch(eFrames)


			{}





			if(canAccessFrame && frameSize > bestFrameSize)


			{


				bestFrame = frames[frameIndex];


				bestFrameSize = frameSize;


			}


		}





		if(bestFrame)


			ReadingToolsCore.cacheDiv.innerHTML = bestFrame.contentWindow.document.body.innerHTML;


	}





	/* remove all scripts that are not ReadingToolsCore */


	var scripts = ReadingToolsCore.cacheDiv.getElementsByTagName('script');


	for(var i = scripts.length-1; i >= 0; i--)


	{


		if(typeof(scripts[i].src) == "undefined" || (scripts[i].src.indexOf('ReadingToolsCore') == -1 && scripts[i].src.indexOf('typekit') == -1))


		{


			scripts[i].parentNode.removeChild(scripts[i]);


		}


	}





	/* Remove all style tags in head (not doing this on IE) - TODO: Why not? */


	var styleTags = ReadingToolsCore.cacheDiv.getElementsByTagName("style");


	for (var st=0;st < styleTags.length; st++)


	{


		styleTags[st].textContent = "";


	}


}





/**


* Get the article title as an H1.


*


* @return void


**/


ReadingToolsCore.getArticleTitle = function()


{


	var articleTitle = document.createElement("DIV");


	articleTitle.id = "ID_RT_ReadTitle";





	if(ReadingToolsCore.isRootPage == true)


	{


		articleTitle.innerHTML = "Not Found";


		return articleTitle;


	}





	var curTitle = "", origTitle = "";





	try


	{


		curTitle = origTitle = document.title;





		if(typeof curTitle != "string")


		{


			/* If they had an element with id "title" in their HTML */


			curTitle = origTitle = ReadingToolsCore.getInnerText(document.getElementsByTagName('title')[0], true);


		}


	}


	catch(e)


	{}





	if(curTitle.match(/ [\|\-] /))


	{


		curTitle = origTitle.replace(/(.*)[\|\-] .*/gi,'$1');





		if(curTitle.split(' ').length < 3)


		{


			curTitle = origTitle.replace(/[^\|\-]*[\|\-](.*)/gi,'$1');


		}


	}


	else if(curTitle.indexOf(': ') !== -1)


	{


		curTitle = origTitle.replace(/.*:(.*)/gi, '$1');





		if(curTitle.split(' ').length < 3)


		{


			curTitle = origTitle.replace(/[^:]*[:](.*)/gi,'$1');


		}


	}


	else if(curTitle.length > 150 || curTitle.length < 15)


	{


		var hOnes = document.getElementsByTagName('h1');


		if(hOnes.length == 1)


		{


			curTitle = ReadingToolsCore.getInnerText(hOnes[0], true);


		}


	}





	curTitle = curTitle.replace( regexps.trimRe, "" );





	if(curTitle.split(' ').length <= 4)


	{


		curTitle = origTitle;


	}





	articleTitle.innerHTML = curTitle;





	return articleTitle;


}





/***


* grabArticle - Using a variety of metrics (content score, classname, element types), find the content that is


*			   most likely to be the stuff a user wants to read. Then return it wrapped up in a div.


*


* @return Element


**/


ReadingToolsCore.getArticleContent = function()


{


	var articleContent = document.createElement("DIV");


	articleContent.id = "ID_RT_ReadContent";





	/**


	* If someone tries to use Readability on a site's root page, give them a warning about usage.


	**/


	if(ReadingToolsCore.isRootPage == true)


	{


		articleContent.innerHTML = "Not Found";


		return articleContent;


	}





	var stripUnlikelyCandidates = ReadingToolsCore.flagIsActive(ReadingToolsCore.FLAG_STRIP_UNLIKELYS);





	/* Turn all double br's into p's */


	/* Note, this is pretty costly as far as processing goes. Maybe optimize later. */


	ReadingToolsCore.cacheDiv.innerHTML = ReadingToolsCore.cacheDiv.innerHTML.replace(regexps.replaceBrsRe, '</p><p>');//.replace(regexps.replaceFontsRe, '<$1span>');





	/**


	* First, node prepping. Trash nodes that look cruddy (like ones with the class name "comment", etc), and turn divs


	* into P tags where they have been used inappropriately (as in, where they contain no other block level elements.)


	*


	* Note: Assignment from index for performance. See http://www.peachpit.com/articles/article.aspx?p=31567&seqNum=5


	* TODO: Shouldn't this be a reverse traversal?


	**/


	var node = null;


	var nodesToScore = [];


	var readingToolsCoreDiv = ReadingToolsCore.cacheDiv.getElementsByTagName('*');


	for(var nodeIndex = 0; (node = readingToolsCoreDiv[nodeIndex]); nodeIndex++)


	{


		/* Remove unlikely candidates */


		if (stripUnlikelyCandidates)


		{


			var unlikelyMatchString = node.className + node.id;


			if (unlikelyMatchString.search(regexps.unlikelyCandidatesRe) !== -1 &&


				unlikelyMatchString.search(regexps.okMaybeItsACandidateRe) == -1 &&


				node.tagName !== "BODY")


			{


				node.parentNode.removeChild(node);


				nodeIndex--;


				continue;


			}


		}





		if (node.tagName == "P" || node.tagName == "TD")


		{


			nodesToScore[nodesToScore.length] = node;


		}





		/* Turn all divs that don't have children block level elements into p's */


		if (node.tagName == "DIV")


		{


			if (node.innerHTML.search(regexps.divToPElementsRe) == -1)


			{


				var newNode = document.createElement('p');


				try


				{


					newNode.innerHTML = node.innerHTML;


					node.parentNode.replaceChild(newNode, node);


					nodeIndex--;


				}


				catch(e)


				{}


			}


			else


			{


				/* EXPERIMENTAL */


				for(var i = 0, il = node.childNodes.length; i < il; i++)


				{


					var childNode = node.childNodes[i];


					if(childNode.nodeType == 3)


					{


						// Node.TEXT_NODE


						var p = document.createElement('p');


						p.innerHTML = childNode.nodeValue;


						p.style.display = 'inline';


						p.className = 'ReadingToolsCore-styled';


						childNode.parentNode.replaceChild(p, childNode);


					}


				}


			}


		}


	}





	/**


	* Loop through all paragraphs, and assign a score to them based on how content-y they look.


	* Then add their score to their parent node.


	*


	* A score is determined by things like number of commas, class names, etc. Maybe eventually link density.


	**/


	var candidates = [];


	for (var pt=0; pt < nodesToScore.length; pt++)


	{


		var parentNode	  = nodesToScore[pt].parentNode;


		var grandParentNode = parentNode.parentNode;


		var innerText	   = ReadingToolsCore.getInnerText(nodesToScore[pt], true);





		/* If this paragraph is less than 25 characters, don't even count it. */


		if(innerText.length < 25 || !grandParentNode)


		{


			continue;


		}





		/* Initialize ReadingToolsCore data for the parent. */


		if(typeof parentNode.ReadingToolsCore == 'undefined')


		{


			ReadingToolsCore.initializeNode(parentNode);


			candidates.push(parentNode);


		}





		/* Initialize ReadingToolsCore data for the grandparent. */


		if(typeof grandParentNode.ReadingToolsCore == 'undefined')


		{


			ReadingToolsCore.initializeNode(grandParentNode);


			candidates.push(grandParentNode);


		}





		var contentScore = 0;





		/* Add a point for the paragraph itself as a base. */


		contentScore++;





		/* Add points for any commas within this paragraph */


		/* For every 100 characters in this paragraph, add another point. Up to 3 points. */


		contentScore += innerText.split(',').length;


		contentScore += Math.min(Math.floor(innerText.length / 100), 3);





		/* Add the score to the parent. The grandparent gets half. */


		parentNode.ReadingToolsCore.contentScore += contentScore;


		grandParentNode.ReadingToolsCore.contentScore += contentScore/2;


	}





	/**


	* After we've calculated scores, loop through all of the possible candidate nodes we found


	* and find the one with the highest score.


	**/


	var topCandidate = null;


	for(var c=0, cl=candidates.length; c < cl; c++)


	{


		/**


		* Scale the final candidates score based on link density. Good content should have a


		* relatively small link density (5% or less) and be mostly unaffected by this operation.


		**/


		candidates[c].ReadingToolsCore.contentScore = candidates[c].ReadingToolsCore.contentScore * (1-ReadingToolsCore.getLinkDensity(candidates[c]));


		if(!topCandidate || candidates[c].ReadingToolsCore.contentScore > topCandidate.ReadingToolsCore.contentScore)


		{


			topCandidate = candidates[c];


		}


	}





	/**


	* If we still have no top candidate, just use the body as a last resort.


	* We also have to copy the body node so it is something we can modify.


	**/


	if (topCandidate == null || topCandidate.tagName == "BODY")


	{


		topCandidate = ReadingToolsCore.cacheDiv;


		topCandidate.innerHTML = ReadingToolsCore.cacheDiv.innerHTML;


//		document.body.appendChild(topCandidate);


		ReadingToolsCore.initializeNode(topCandidate);


		return articleContent;


	}





	/**


	* Now that we have the top candidate, look through its siblings for content that might also be related.


	* Things like preambles, content split by ads that we removed, etc.


	**/


	var siblingScoreThreshold = Math.max(10, topCandidate.ReadingToolsCore.contentScore * 0.2);


	var siblingNodes = topCandidate.parentNode.childNodes;





	for(var s=0, sl=siblingNodes.length; s < sl; s++)


	{


		var siblingNode = siblingNodes[s];


		var append = false;





		if(siblingNode == topCandidate)


		{


			append = true;


		}





		var contentBonus = 0;


		/* Give a bonus if sibling nodes and top candidates have the example same classname */


		if(siblingNode.className == topCandidate.className && topCandidate.className != "")


		{


			contentBonus += topCandidate.ReadingToolsCore.contentScore * 0.2;


		}





		if(typeof siblingNode.ReadingToolsCore != 'undefined' && (siblingNode.ReadingToolsCore.contentScore+contentBonus) >= siblingScoreThreshold)


		{


			append = true;


		}





		if(siblingNode.nodeName == "P")


		{


			var linkDensity = ReadingToolsCore.getLinkDensity(siblingNode);


			var nodeContent = ReadingToolsCore.getInnerText(siblingNode, true);


			var nodeLength  = nodeContent.length;





			if(nodeLength > 80 && linkDensity < 0.25)


			{


				append = true;


			}


			else if(nodeLength < 80 && linkDensity == 0 && nodeContent.search(/\.( |$)/) !== -1)


			{


				append = true;


			}


		}





		if(append)


		{


			var nodeToAppend = null;


			if(siblingNode.nodeName != "DIV" && siblingNode.nodeName != "P")


			{


				/* We have a node that isn't a common block level element, like a form or td tag. Turn it into a div so it doesn't get filtered out later by accident. */





				nodeToAppend = document.createElement('div');


				try


				{


					nodeToAppend.id = siblingNode.id;


					nodeToAppend.innerHTML = siblingNode.innerHTML;


				}


				catch(e)


				{


					nodeToAppend = siblingNode;


					s--;


					sl--;


				}


			}


			else


			{


				nodeToAppend = siblingNode;


				s--;


				sl--;


			}





			/* To ensure a node does not interfere with ReadingToolsCore styles, remove its classnames */


			nodeToAppend.className = "";





			/* Append sibling and subtract from our list because it removes the node when you append to another node */


			articleContent.appendChild(nodeToAppend);


		}


	}





	return articleContent;


}





/**


* Prepare the article node for display. Clean out any inline styles,


* iframes, forms, strip extraneous <p> tags, etc.


*


* @param Element


* @return void


**/


ReadingToolsCore.prepArticle = function(articleContent)


{


	ReadingToolsCore.cleanStyles(articleContent);


//	ReadingToolsCore.killBreaks(articleContent);





	/* Clean out junk from the article content */


	ReadingToolsCore.clean(articleContent, "form");


	ReadingToolsCore.clean(articleContent, "object");


	ReadingToolsCore.clean(articleContent, "h1");


	/**


	* If there is only one h2, they are probably using it


	* as a header and not a subheader, so remove it since we already have a header.


	***/


	if(articleContent.getElementsByTagName('h2').length == 1)


	{


		ReadingToolsCore.clean(articleContent, "h2");


	}





	ReadingToolsCore.clean(articleContent, "iframe");


	ReadingToolsCore.cleanHeaders(articleContent);





	/* Do these last as the previous stuff may have removed junk that will affect these */


	ReadingToolsCore.cleanConditionally(articleContent, "table");


	ReadingToolsCore.cleanConditionally(articleContent, "ul");


	ReadingToolsCore.cleanConditionally(articleContent, "div");





	/* Remove extra paragraphs */


	var articleParagraphs = articleContent.getElementsByTagName('p');


	for(var i = articleParagraphs.length-1; i >= 0; i--)


	{


		var imgCount	= articleParagraphs[i].getElementsByTagName('img').length;


		var embedCount  = articleParagraphs[i].getElementsByTagName('embed').length;


		var objectCount = articleParagraphs[i].getElementsByTagName('object').length;





		if(imgCount == 0 && embedCount == 0 && objectCount == 0 && ReadingToolsCore.getInnerText(articleParagraphs[i], false) == '')


		{


			articleParagraphs[i].parentNode.removeChild(articleParagraphs[i]);


		}


	}





	try


	{


		articleContent.innerHTML = articleContent.innerHTML.replace(/<br[^>]*>\s*<p/gi, '<p');


	}


	catch (e)


	{}


}





/**


* For easier reading, convert this document to have footnotes at the bottom rather than inline links.


* @see http://www.roughtype.com/archives/2010/05/experiments_in.php


*


* @return void


**/


ReadingToolsCore.addFootnotes = function(articleContent)


{


//		var footnotesWrapper = document.createElement('div');


//		footnotesWrapper.innerHTML = "<h3>References</h3>";


//


//		var articleFootnotes = document.createElement('ol');


//		footnotesWrapper.appendChild(articleFootnotes);





	var articleLinks = articleContent.getElementsByTagName('a');





//	var linkCount = 0;


	for (var i = 0; i < articleLinks.length; i++)


	{


		var articleLink = articleLinks[i];


//			var footnoteLink = articleLink.cloneNode(true);


//			var refLink = document.createElement('a');


//			var footnote = document.createElement('li');


//			var linkDomain = footnoteLink.host ? footnoteLink.host : document.location.host;


//			var linkText = ReadingToolsCore.getInnerText(articleLink, true);


//


//			if(articleLink.className && articleLink.className.indexOf('ReadingToolsCore-DoNotFootnote') !== -1 || linkText.match(regexps.skipFootnoteLinkRe))


//			{


//				continue;


//			}


//


//			linkCount++;


//


//			/** Add a superscript reference after the article link */


//			refLink.href	  = '#ReadingToolsCoreFootnoteLink-' + linkCount;


//			refLink.innerHTML = '<small><sup>[' + linkCount + ']</sup></small>'


//			refLink.className = 'ReadingToolsCore-DoNotFootnote';


//			refLink.style.color = 'inherit';


//


//			if(articleLink.parentNode.lastChild == articleLink)


//			{


//				articleLink.parentNode.appendChild(refLink);


//			}


//			else


//			{


//				articleLink.parentNode.insertBefore(refLink, articleLink.nextSibling);


//			}





		articleLink.style.color = 'inherit';


		articleLink.style.textDecoration = "none";


		articleLink.removeAttribute("href");


//			articleLink.name = 'ReadingToolsCoreLink-' + linkCount;


//


//			footnote.innerHTML = "<small><sup><a href='#ReadingToolsCoreLink-" + linkCount + "' title='Jump to Link in Article'>^</a></sup></small> ";


//


//			footnoteLink.innerHTML = (footnoteLink.title ? footnoteLink.title : linkText);


//			footnoteLink.name = 'ReadingToolsCoreFootnoteLink-' + linkCount;


//


//			footnote.appendChild(footnoteLink);


//			footnote.innerHTML = footnote.innerHTML + "<small> (" + linkDomain + ")</small>";


//


//			articleFootnotes.appendChild(footnote);


	}





//		if(linkCount > 0)


//		{


//			articleContent.appendChild(footnotesWrapper);


//		}


}





/**


* Initialize a node with the ReadingToolsCore object. Also checks the


* className/id for special names to add to its score.


*


* @param Element


* @return void


**/


ReadingToolsCore.initializeNode = function(node)


{


	node.ReadingToolsCore = {"contentScore": 0};





	switch(node.tagName)


	{


	case 'DIV':


		node.ReadingToolsCore.contentScore += 5;


		break;





	case 'PRE':


	case 'TD':


	case 'BLOCKQUOTE':


		node.ReadingToolsCore.contentScore += 3;


		break;





	case 'ADDRESS':


	case 'OL':


	case 'UL':


	case 'DL':


	case 'DD':


	case 'DT':


	case 'LI':


	case 'FORM':


		node.ReadingToolsCore.contentScore -= 3;


		break;





	case 'H1':


	case 'H2':


	case 'H3':


	case 'H4':


	case 'H5':


	case 'H6':


	case 'TH':


		node.ReadingToolsCore.contentScore -= 5;


		break;


	}





	node.ReadingToolsCore.contentScore += ReadingToolsCore.getClassWeight(node);


}





/**


* Get the inner text of a node - cross browser compatibly.


* This also strips out any excess whitespace to be found.


*


* @param Element


* @return string


**/


ReadingToolsCore.getInnerText = function(e, normalizeSpaces)


{


	var textContent = e.textContent.replace( regexps.trimRe, "" );


	if(normalizeSpaces)


	{


		return textContent.replace( regexps.normalizeRe, " ");


	}


	else


	{


		return textContent;


	}


}





/**


* Get the number of times a string s appears in the node e.


*


* @param Element


* @param string - what to split on. Default is ","


* @return number (integer)


**/


ReadingToolsCore.getCharCount = function(e, s)


{


	s = s || ",";


	return ReadingToolsCore.getInnerText(e, true).split(s).length-1;


}





/**


* Get the density of links as a percentage of the content


* This is the amount of text that is inside a link divided by the total text in the node.


*


* @param Element


* @return number (float)


**/


ReadingToolsCore.getLinkDensity = function(e)


{


	var links = e.getElementsByTagName("a");


	var textLength = ReadingToolsCore.getInnerText(e, true).length;


	var linkLength = 0;


	for(var i=0, il=links.length; i<il;i++)


	{


		linkLength += ReadingToolsCore.getInnerText(links[i], true).length;


	}





	return linkLength / textLength;


}





/**


* Get an elements class/id weight. Uses regular expressions to tell if this


* element looks good or bad.


*


* @param Element


* @return number (Integer)


**/


ReadingToolsCore.getClassWeight = function(e)


{


	if(!ReadingToolsCore.flagIsActive(ReadingToolsCore.FLAG_WEIGHT_CLASSES))


	{


		return 0;


	}





	var weight = 0;





	/* Look for a special classname */


	if (e.className != "")


	{


		if(e.className.search(regexps.negativeRe) !== -1)


		{


			weight -= 25;


		}





		if(e.className.search(regexps.positiveRe) !== -1)


		{


			weight += 25;


		}


	}





	/* Look for a special ID */


	if (typeof(e.id) == 'string' && e.id != "")


	{


		if(e.id.search(regexps.negativeRe) !== -1)


		{


			weight -= 25;


		}





		if(e.id.search(regexps.positiveRe) !== -1)


		{


			weight += 25;


		}


	}





	return weight;


}





/**


* Remove extraneous break tags from a node.


*


* @param Element


* @return void


**/


ReadingToolsCore.killBreaks = function(e)


{


	try


	{


		e.innerHTML = e.innerHTML.replace(regexps.killBreaksRe,'<br>');


	}


	catch (eBreaks)


	{}


}





/**


* Clean a node of all elements of type "tag".


* (Unless it's a youtube/vimeo video. People love movies.)


*


* @param Element


* @param string tag to clean


* @return void


**/


ReadingToolsCore.clean = function(e, tag)


{


	var targetList = e.getElementsByTagName( tag );


	var isEmbed = (tag == 'object' || tag == 'embed');





	for (var y=targetList.length-1; y >= 0; y--)


	{


		/* Allow youtube and vimeo videos through as people usually want to see those. */


		if(isEmbed)


		{


			var attributeValues = "";


			for (var i=0, il=targetList[y].attributes.length; i < il; i++)


			{


				attributeValues += targetList[y].attributes[i].value + '|';


			}





			/* First, check the elements attributes to see if any of them contain youtube or vimeo */


			if (attributeValues.search(regexps.videoRe) !== -1)


			{


				continue;


			}





			/* Then check the elements inside this element for the same. */


			if (targetList[y].innerHTML.search(regexps.videoRe) !== -1)


			{


				continue;


			}


		}





		targetList[y].parentNode.removeChild(targetList[y]);


	}


}





/**


* Remove the style attribute on every e and under.


* TODO: Test if getElementsByTagName(*) is faster.


*


* @param Element


* @return void


**/


ReadingToolsCore.cleanStyles = function(e)


{


	e = e || document;


	var cur = e.firstChild;





	if(!e)


		return;





	// Remove any root styles, if we're able.


	if(typeof e.removeAttribute == 'function' && e.className != 'ReadingToolsCore-styled' && e.tagName != "FONT")


	{


		e.removeAttribute('style');


	}





	// Go until there are no more child nodes


	while ( cur !== null )


	{


		if ( cur.nodeType == 1 )


		{


			// Remove style attribute(s) :


			if(cur.className != "ReadingToolsCore-styled" && cur.tagName != "FONT")


			{


				cur.removeAttribute("style");


			}


			ReadingToolsCore.cleanStyles( cur );


		}


		cur = cur.nextSibling;


	}


}





/**


* Clean an element of all tags of type "tag" if they look fishy.


* "Fishy" is an algorithm based on content length, classnames, link density, number of images & embeds, etc.


*


* @return void


**/


ReadingToolsCore.cleanConditionally = function(e, tag)


{


	var tagsList = e.getElementsByTagName(tag);


	var curTagsLength = tagsList.length;





	/**


	* Gather counts for other typical elements embedded within.


	* Traverse backwards so we can remove nodes at the same time without effecting the traversal.


	*


	* TODO: Consider taking into account original contentScore here.


	**/


	for (var i=curTagsLength-1; i >= 0; i--)


	{


		var weight = ReadingToolsCore.getClassWeight(tagsList[i]);


		var contentScore = (typeof tagsList[i].ReadingToolsCore != 'undefined') ? tagsList[i].ReadingToolsCore.contentScore : 0;





		if(weight+contentScore < 0)


		{


			tagsList[i].parentNode.removeChild(tagsList[i]);


		}


		else if ( ReadingToolsCore.getCharCount(tagsList[i],',') < 10)


		{


			/**


			* If there are not very many commas, and the number of


			* non-paragraph elements is more than paragraphs or other ominous signs, remove the element.


			**/


			var p = tagsList[i].getElementsByTagName("p").length;


			var img = tagsList[i].getElementsByTagName("img").length;


			var li = tagsList[i].getElementsByTagName("li").length-100;


			var input = tagsList[i].getElementsByTagName("input").length;





			var embedCount = 0;


			var embeds = tagsList[i].getElementsByTagName("embed");


			for(var ei=0,il=embeds.length; ei < il; ei++)


			{


				if (embeds[ei].src.search(regexps.videoRe) == -1)


				embedCount++;


			}





			var max = 0;


			for(var count=0; count < img; count++)


			{


				var imgElm = tagsList[i].getElementsByTagName("img")[count];


				// not remove. this line is reset src of the image.


				imgElm.src = imgElm.src;





				var maxLength = (imgElm.style.height > imgElm.style.width) ? imgElm.height : imgElm.width;


				if (maxLength > max)


				{


					max = maxLength;


				}


			}





			var linkDensity = ReadingToolsCore.getLinkDensity(tagsList[i]);


			var contentLength = ReadingToolsCore.getInnerText(tagsList[i], true).length;


			var toRemove = false;





			if ( img > p )


			{


				toRemove = true;


			}


			else if(li > p && tag != "ul" && tag != "ol")


			{


				toRemove = true;


			}


			else if( input > Math.floor(p/3) )


			{


			toRemove = true;


			}


			else if(contentLength < 25 && (img == 0 || img > 2) )


			{


			toRemove = true;


			}


			else if(weight < 25 && linkDensity > 0.2)


			{


			toRemove = true;


			}


			else if(weight >= 25 && linkDensity > 0.5)


			{


			toRemove = true;


			}


			else if((embedCount == 1 && contentLength < 75) || embedCount > 1)


			{


			toRemove = true;


			}





			if(max > 200)


			{


				toRemove = false;


			}





			if(toRemove)


			{


				tagsList[i].parentNode.removeChild(tagsList[i]);


			}


		}


	}


}





/**


* Clean out spurious headers from an Element. Checks things like classnames and link density.


*


* @param Element


* @return void


**/


ReadingToolsCore.cleanHeaders = function(e)


{


	for (var headerIndex = 1; headerIndex < 7; headerIndex++)


	{


		var headers = e.getElementsByTagName('h' + headerIndex);


		for (var i=headers.length-1; i >=0; i--)


		{


			if (ReadingToolsCore.getClassWeight(headers[i]) < 0 || ReadingToolsCore.getLinkDensity(headers[i]) > 0.33)


			{


				headers[i].parentNode.removeChild(headers[i]);


			}


		}


	}


}





ReadingToolsCore.flagIsActive = function(flag)


{


	return (ReadingToolsCore.flags & flag) > 0;


}





ReadingToolsCore.addFlag = function(flag)


	{


	ReadingToolsCore.flags = ReadingToolsCore.flags | flag;


}





ReadingToolsCore.removeFlag = function(flag)


{


	ReadingToolsCore.flags = ReadingToolsCore.flags & ~flag;


}





ReadingToolsCore.getReadInnerHTML = function()


{


	return document.getElementById("ID_RT_ReadInner").innerHTML;


}





ReadingToolsCore.init();


var temp = ReadingToolsCore.getReadInnerHTML(); 


temp.toString();


//temp.toUTF8String();


 encode_utf8=function( temp ){


return unescape( encodeURIComponent( temp ) );


}


var temp1=encode_utf8(temp);


temp1.toString();


temp.toString();


