//  TvApi.h
//
//  Konfabulator
//
//  Copyright (c) 2008-2009 Yahoo! Inc. All rights reserved.
//
//-----------------------------------------------------------------------------
/*
 	1.7.1 PP
 		Added ExtendedError to ErrorCode enumeration
 		Added getExtendedError to MasterControl class (with default implementation)
 		Added setExtraParameters to NetworkInput class (with default implementation)

	1.7 AN
		Added setCallbacks() to MasterControl.

	1.6	AN
		Removed getBufferMaxDuration().
		Added getBufferedPercentage()
		This function returns an integer in the range 0 to 100 (inclusive) to represent the percentage
		of the buffer that is full. This value can be calculated using the total size of the buffer and does
		not depend on the time information from the stream.
		If there is no buffer or buffering is not supported, this function should return 100.

 	1.5.1	PP

 			Added KON_EXPORTABLE decoration to all classes - this is to resolve issues with weak symbols
 			and dynamic casts across shared objects.

 			Changed #define API_VERSION to a string and renamed it.

    1.5    	AN

			Modified buffer(). Takes only duration to buffer in msecs.
			If duration is 0, then buffer data for a duration determined by the OEM.
			If duration is > 0, then buffer data for that duration.
			Return true if data can be buffered and be made available for display immediately after play() is called.
			Return false if data cannot be buffered until play() is called. In this case use the input parameter as the
			duration of data that will be buffered until display of the media stream on the screen.

			Added getWidth() and getHeight() to FileInput and NetworkInput.

			Added doesResumePlay to PlaybackControlSupport. doesResumePlay indicates the ability to start buffering from a specific position
			in the stream.

			Other notes :

			Bitrate URL Switch
			------------------
			When the uri on a  NetworkInput is changed, a path attempts to switch display to the new uri without any noticeable interruption.
			Do not stop() the PlaybackControl in the Path. If the PlaybackControl needs to buffer data, then freeze the current frame and set
			the state to Buffering until playback can be restored. This is how bitrate url switching is handled by the TV API interface.

			Stopping a PlaybackControl
			--------------------------
			A path between a NetworkInput or FileInput and a ScreenOutput manages a unique media stream. If you stop() a PlaybackControl, it may mean
			that the path is either being reused for a new media stream (and not for switching to an uri for a different bitrate) or it is being prepared
			for teardown. If the input in the path is not changed, and stop() is followed by a play(), the media stream should resume from the current
			timeIndex.

			PlaybackControlBufferChanged Callback
			-------------------------------------
			The PlaybackControlBufferChanged callback should be fired when the amount of buffered data passes the half-capacity mark.

 	1.4  	PP

 			Added RemovableMedia class
 			Added Callbacks::RemovableMediaConnected, Callbacks::RemovableMediaDisconnected and
 				Callbacks::RemovableMediaDisconnectedSafely
 			Added System::getRemovableMedia
 			Added WirelessNetworkInterface::disconnect
 			Added CertificateStore class
 			Added System::getCertificateStore

 			AN

 			Added overloaded PlaybackControl::buffer( long ). Takes in number of msecs to buffer. When this function is called, the player will not play until
 			the buffer is full.
 			buffer() without an input will buffer a default number of msecs. play() can be called while this buffer is being filled.
 			Added doesBuffer to enum PlaybackControl::doesSupport

 	1.3    	AN

		   	Added buffer(), getBufferedDuration(), getBufferMaxDuration() and getIsBufferFull() to PlaybackControl
		   	Added PlaybackControlBufferChanged to Callbacks
		   	Added getHttpErrorCode() to NetworkInput
		   	Added getTimeZoneGMTOffset() to System
		  	Added ViewportMode, setViewPortMode and getViewPortMode to ScreenOutput

	1.2 	PP

			Added NetworkInterface class
			Added WirelessNetworkInterface class
			Added System::getNetworkInterfaces
			Added StringList typedef
			Added List typedefs for NetworkInterface, WirelessNetwork, Program, Channel,
				Input, Output and Path
			Removed const from return type for MasterControl::getOutputs, getInputs,
				getConnectedInputs and getPaths
			Power and SoftwareUpdate are now classes instead of structs
			All getters are now declared const
			All pointers have been changed to reference counted smart pointers
				Except for callbacks where they remain as pointers
			Entry point has been changed to use refs

*/
//-----------------------------------------------------------------------------

#ifndef __TVAPI_H__
#define __TVAPI_H__

#include <list>
#include <string>

#include "KFCommon.h"
#include "KFSmartPointer.h"

namespace TV
{

#define KF_TV_API_VERSION "1.7.1"

struct Rectangle
{
	double	x;
	double	y;
	double	width;
	double	height;
};

struct ChannelNum
{
	int MajorChannelNum;
	int MinorChannelNum;
};

enum AspectRatio
{
	AspectNotAvailable = 0,
	AspectOther,
	AspectSquare,
	Aspect4x3,
	Aspect16x9
};

enum DisplayType
{
	DisplayTypeNotAvailable = 0,
	Plasma,
	DLP,
	LCD,
	OLED,
	Other
};

enum Broadcast
{
	BroadcastTypeNotAvailable = 0,
	BroadcastOther,
	DVBT,
	ATSC
};

enum ErrorCode
{
	NoError = 0,
	FileNotFound,
	NetworkUnavailable,
	FileReadError,
	Unknown,
	ExtendedError
};


enum PowerStatus
{
	PowerStatusNotAvailable = 0,
	PowerOff,
	PowerOn,
	Sleep,
	Wake
};


enum NetworkStatus
{
	NetworkStatusNotAvailable = 0,
	Disconnected,
	Connected
};

enum VChipStatus
{
	NotInstalled = 0,
	Enabled,
	NotEnabled
};

//-------------------------------------------------------------
// Utility types
//-------------------------------------------------------------

typedef std::list<std::string> StringList;

typedef YWE::Retainable Retainable;

//-------------------------------------------------------------
// Forward declarations (including ref types)
//-------------------------------------------------------------

FORWARD_DECLARE_REF(Callbacks)
FORWARD_DECLARE_REF(MasterControl)
FORWARD_DECLARE_REF(Control)
FORWARD_DECLARE_REF(PlaybackControl)
FORWARD_DECLARE_REF(TunerControl)
FORWARD_DECLARE_REF(NetworkInterface)
FORWARD_DECLARE_REF(WirelessNetworkInterface)
FORWARD_DECLARE_REF(System)
FORWARD_DECLARE_REF(Path)
FORWARD_DECLARE_REF(Audio)
FORWARD_DECLARE_REF(Input)
FORWARD_DECLARE_REF(TunerInput)
FORWARD_DECLARE_REF(FileInput)
FORWARD_DECLARE_REF(NetworkInput)
FORWARD_DECLARE_REF(Output)
FORWARD_DECLARE_REF(ScreenOutput)
FORWARD_DECLARE_REF(Channel)
FORWARD_DECLARE_REF(Program)
FORWARD_DECLARE_REF(Power)
FORWARD_DECLARE_REF(SoftwareUpdate)
FORWARD_DECLARE_REF(RemovableMedia)
FORWARD_DECLARE_REF(CertificateStore)

//-------------------------------------------------------------
//  Entry point
//-------------------------------------------------------------

typedef MasterControlRef ( * ENTRY_POINT )( CallbacksRef );

//-------------------------------------------------------------
// Callbacks
//-------------------------------------------------------------

class KON_EXPORTABLE Callbacks : public Retainable
{
public:

    // MasterControl

    virtual void NetworkStatusChanged( MasterControl * ) = 0;
    virtual void ErrorOccurred( MasterControl * ) = 0;

    // Power

    virtual void PowerStatusChanged( Power * ) = 0;

    // SoftwareUpdate

    virtual void SoftwareUpdate( TV::SoftwareUpdate * ) = 0;

    // Path

    virtual void PathDestroyed( Path * ) = 0;
    virtual void PathChanged( Path * ) = 0;

    // System

    virtual void AspectRatioChanged( System * ) = 0;
    virtual void MenuLanguageChanged( System * ) = 0;

    // Audio

    virtual void VolumeChanged( Audio * ) = 0;

    // TunerInput

    virtual void ChannelLineupChanged( TunerInput * ) = 0;

    // Input

    virtual void NewInputDetected( Input * ) = 0;
    virtual void InputDisconnected( Input * ) = 0;
    virtual void InputStateChanged( Input * ) = 0;

    // ScreenOutput

    virtual void BoundsChanged( ScreenOutput * ) = 0;

    // PlaybackControl

    virtual void PlaybackControlStateChanged( PlaybackControl * ) = 0;
    virtual void PlaybackControlTimeIndexChanged( PlaybackControl * ) = 0;
    virtual void PlaybackControlBufferChanged( PlaybackControl * ) = 0;

    // TunerControl

    virtual void ChannelChanged( TunerControl * ) = 0;
    virtual void ProgramChanged( TunerControl * ) = 0;

    // RemovableMedia

    virtual void RemovableMediaConnected( RemovableMedia * ) = 0;
    virtual void RemovableMediaDisconnectedSafely( RemovableMedia * ) = 0;
    virtual void RemovableMediaDisconnected( RemovableMedia * ) = 0;
};

//-------------------------------------------------------------
//  Power
//-------------------------------------------------------------

class KON_EXPORTABLE Power : public Retainable
{
public:

	virtual PowerStatus getCurrentState() const = 0;
	virtual PowerStatus getPreviousState() const = 0;

protected:

	virtual ~Power() {}
};

//-------------------------------------------------------------
//  SoftwareUpdate
//-------------------------------------------------------------

class KON_EXPORTABLE SoftwareUpdate : public Retainable
{
public:

	virtual std::string getCurrentVersion() const = 0;
	virtual std::string getUpdateVersion() const = 0;
	virtual std::string getURI() const = 0;
	virtual std::string getDescription() const = 0;
	virtual bool		isCritical() const = 0;

protected:

	virtual ~SoftwareUpdate() {}
};

//-------------------------------------------------------------
//  NetworkInterface
//-------------------------------------------------------------

class KON_EXPORTABLE NetworkInterface : public Retainable
{
public:

	typedef std::list<NetworkInterfaceRef> List;

	enum Type
	{
		Wired,
		Wireless
	};

	virtual NetworkInterface::Type 	getType() const = 0;

	virtual std::string 			getIdentifier() const = 0;

	virtual std::string				getLabel() const = 0;

	virtual bool 					isDHCPEnabled() const = 0;
	virtual bool					enableDHCP() = 0;
	virtual bool					disableDHCP() = 0;

	virtual std::string				getDynamicIPAddress() const = 0;

	virtual std::string 			getStaticIPAddress() const = 0;
	virtual bool					setStaticIPAddress( const std::string & inStaticIPAddress ) = 0;

	virtual std::string				getSubnetMask() const = 0;
	virtual bool					setSubnetMask( const std::string & inSubnetMask ) = 0;

	virtual std::string				getDefaultGateway() const = 0;
	virtual bool					setDefaultGateway( const std::string & inDefaultGateway ) = 0;

	virtual std::string				getMACAddress() const = 0;

	virtual StringList				getDNSAddresses() const = 0;
	virtual bool					setDNSAddresses( const StringList & inDNSAddresses ) = 0;

protected:

	virtual 						~NetworkInterface() {}
};

//-------------------------------------------------------------
//  WirelessNetworkInterface
//-------------------------------------------------------------

class KON_EXPORTABLE WirelessNetworkInterface : public NetworkInterface
{
public:

	enum SignalStrength
	{
		NoSignal,
		VeryLow,
		Low,
		Medium,
		High,
		VeryHigh
	};

	enum WirelessSecurity
	{
		NoSecurity,
		WEP,
		WPA,
		WPA2,
		WPAPSK,
		WPA2PSK
	};

	struct WirelessNetwork
	{
		std::string			SSID;
		SignalStrength		signalStrength;
		WirelessSecurity	wirelessSecurity;

		typedef std::list<WirelessNetwork> List;
	};

	virtual WirelessNetwork::List		getAvailableWirelessNetworks() const = 0;

	virtual bool						isConnected() const = 0;

	virtual WirelessNetwork				getConnectedWirelessNetwork() const = 0;

	virtual bool						connectToWirelessNetwork(
											const std::string & inSSID ,
											const std::string & inKeyOrPassPhrase ,
											const WirelessSecurity inWirelessSecurity ) = 0;

	virtual bool						disconnect() = 0;

protected:

	virtual ~WirelessNetworkInterface() {}
};

//-------------------------------------------------------------
//  RemovableMedia
//-------------------------------------------------------------

class KON_EXPORTABLE RemovableMedia : public Retainable
{
public:

	typedef std::list<RemovableMediaRef>	List;

	enum Type
	{
		UknownType,
		USB,
		IEEE1394,
		Optical,
		Flash,
		Network
	};

	virtual std::string				getIdentifier() const = 0;
	virtual std::string				getLabel() const = 0;
	virtual std::string				getPath() const = 0;

	virtual RemovableMedia::Type	getType() const = 0;

	virtual bool					isReadOnly() const = 0;

	virtual unsigned long long		getFreeBytes() const = 0;

	virtual bool					eject() = 0;

protected:

	virtual ~RemovableMedia() {}
};

//-------------------------------------------------------------
//	CertificateStore
//-------------------------------------------------------------

class KON_EXPORTABLE CertificateStore : public Retainable
{
public:

	virtual std::string 			getPEMCertificate( const std::string & inName ) const = 0;
	virtual std::string				getPEMPrivateKey( const std::string & inName ) const = 0;

protected:

	virtual ~CertificateStore() {}
};

//-------------------------------------------------------------
//  System
//-------------------------------------------------------------

class KON_EXPORTABLE System : public Retainable
{
public:

	virtual std::string				getOEM() const = 0;
	virtual std::string				getDeviceDescription() const = 0;
	virtual std::string				getDeviceId() const = 0;
	virtual std::string				getDeviceVersion() const = 0;
	virtual std::string				getDeviceSoftwareVersion() const = 0;
	virtual long					getClock() const = 0;
	virtual AspectRatio				getAspectRatio() const = 0;
	virtual std::string				getMenuLanguage() const = 0;
	virtual std::string				getCountry() const = 0;
	virtual PowerRef	    		getPower() const = 0;
	virtual std::string				getDeviceModel() const = 0;
	virtual std::string				getDeviceClass() const = 0;

	virtual std::string				getDeviceBrand() const = 0;
	virtual std::string				getNetworkType() const = 0;
	virtual std::string				getWirelessSignalStrength() const = 0;

	virtual	DisplayType				getDeviceDisplayType() const = 0;

	virtual	NetworkStatus			getNetworkStatus( std::string inNetworkType ) const = 0;

	virtual	VChipStatus				getVChipStatus() const = 0;
	virtual	std::string				getVChipRatingSystem() const = 0;
	virtual	std::string				getLocationCode() const = 0;
	virtual	std::string				getDeviceInfo() const = 0;

	virtual NetworkInterface::List	getNetworkInterfaces() const { return NetworkInterface::List(); }

	virtual RemovableMedia::List	getRemovableMedia() const { return RemovableMedia::List(); }

	virtual CertificateStoreRef		getCertificateStore() const { return NULL_REF; }

	virtual float					getTimeZoneGMTOffset() const = 0;

protected:

	virtual ~System() {}
};

//-------------------------------------------------------------
//  Channel
//-------------------------------------------------------------

class KON_EXPORTABLE Channel : public Retainable
{
public:

	typedef std::list<ChannelRef> List;

	virtual std::string	getShortName() const = 0;
	virtual ChannelNum	getChannelNumber() const = 0;

protected:

	virtual ~Channel() {}
};

//-------------------------------------------------------------
//  Program
//-------------------------------------------------------------

class KON_EXPORTABLE Program : public Retainable
{
public:

	typedef std::list<ProgramRef> List;

	virtual std::string getTitle() const = 0;
	virtual std::string getShortDescription() const = 0;
	virtual std::string	getGenre() const = 0;
	virtual std::string	getPrimaryAudioLanguage() const = 0;
	virtual long		getStartTime() const = 0;
	virtual long		getDuration() const = 0;
	virtual std::string	getRatingSystem() const = 0;
	virtual std::string	getRatingLevel() const = 0;
	virtual bool		isClosedCaptionAvailable() const = 0;
	virtual bool		isSAPAvailable() const = 0;
	virtual bool		isSubTitleAvailable() const = 0;

protected:

	virtual ~Program() {}
};

//-------------------------------------------------------------
//  INPUT
//-------------------------------------------------------------

class KON_EXPORTABLE Input : public Retainable
{
public:

	typedef std::list<InputRef> List;

	enum InputType
	{
		TypeNotAvailable = 0,
		TypeOther,
		TypeTuner,
		TypeHDMI,
		TypeComponent,
		TypeSVideo,
		TypeComposite,
		TypeNetwork,
		TypeFixedStorage,
		TypeRemovableStorage
	};

	//Standard Interface

	virtual InputType	getType() const = 0;
	virtual std::string	getLabel() const = 0;
	virtual bool		isConnected() const = 0;

protected:

	virtual ~Input() {}
};

class KON_EXPORTABLE TunerInput : public Input
{
public:

	enum TunerType
	{
		TypeNotAvailable = 0,
		TunerBuiltIn,
		TunerSerial,
		TunerIR,
		TunerOther
	};


	virtual TunerType		getTunerType() const = 0;
	virtual int				getNumberOfChannels() const = 0;
	virtual Channel::List	getChannels( int from, int to ) const = 0;

protected:

	virtual ~TunerInput() {}
};

class KON_EXPORTABLE FileInput : public Input
{
public:

	enum State
	{
		NONE = 0,
		FILE_NOT_FOUND,
		FILE_FOUND,
		READ_ERROR,
		FORBIDDEN,
		ERROR
	};

	virtual bool		setURI( const std::string& inUri ) = 0;
	virtual std::string	getURI() const = 0;

	virtual InputType	getType() const = 0;
	virtual State		getState() const = 0;

	virtual int			getWidth() const = 0;
	virtual int			getHeight() const = 0;

protected:

	virtual ~FileInput() {}
};

class KON_EXPORTABLE NetworkInput : public Input
{
public:

	enum State
	{
		NONE = 0,
		FILE_NOT_FOUND,
		FILE_FOUND,
		READ_ERROR,
		FORBIDDEN,
		ERROR
	};

	virtual bool		setURI( const std::string& inUri ) = 0;
	virtual std::string	getURI() const = 0;
	virtual State		getState() const = 0;

	virtual InputType	getType() const = 0;

	virtual int			getHttpErrorCode() const = 0;

	virtual int			getWidth() const = 0;
	virtual int			getHeight() const = 0;
	
	virtual bool		setExtraParameters( const StringList & inParameters ) { return true; } 

protected:

	virtual ~NetworkInput() {}
};

//-------------------------------------------------------------
//  OUTPUT
//-------------------------------------------------------------

class KON_EXPORTABLE Output : public Retainable
{
public:

	typedef std::list<OutputRef> List;

	enum OutputType
	{
		TypeNotAvailable = 0,
		TypeNull,
		TypeScreen
	};

	//Standard Interface
	virtual	OutputType	getType() const = 0;
	virtual std::string	getLabel() const = 0;

protected:

	virtual ~Output() {}
};

class KON_EXPORTABLE ScreenOutput : public Output
{
public:

	enum ViewportMode
	{
		ModeNotAvailable = 0,
		ModeOverlay,
		ModeViewport,
		ModeCenteredOverlay
	};

	virtual Rectangle	getBounds() const = 0;
	virtual bool		setBounds( Rectangle inRect ) = 0;

	virtual ViewportMode getViewportMode() const = 0;
	virtual void setViewportMode( ViewportMode inViewportMode ) = 0;

protected:

	virtual ~ScreenOutput() {}
};

//-------------------------------------------------------------
//  Audio
//-------------------------------------------------------------

class KON_EXPORTABLE Audio : public Retainable
{
public:

	virtual bool	setVolume( int value ) = 0;
	virtual int		getVolume() const = 0;
	virtual bool	mute( bool shouldMute ) = 0;
	virtual bool	getMuteState() const = 0;

protected:

	virtual ~Audio() {}
};

//-------------------------------------------------------------
//  PATH
//-------------------------------------------------------------

class KON_EXPORTABLE Path : public Retainable
{
public:

	typedef std::list<PathRef> List;

	//Inputs
	virtual bool		setInput( InputRef inInput ) = 0;
	virtual InputRef	getInput() const = 0;

	//Outputs
	virtual bool		setOutput( OutputRef inOutput ) = 0;
	virtual OutputRef	getOutput() const = 0;

	//Control Access
	virtual ControlRef	getControl() const = 0;

protected:

	virtual ~Path() {}
};

//-------------------------------------------------------------
//  MasterControl
//-------------------------------------------------------------

class KON_EXPORTABLE MasterControl : public Retainable
{
public:

    MasterControl( CallbacksRef inCallbacks ) : fCallbacks( inCallbacks ) {}

	//Path Control
	virtual PathRef			createPath( InputRef inInput, OutputRef inOutput ) = 0;
	virtual Path::List		getPaths() const = 0;
	virtual bool			destroyPath( PathRef inPath ) = 0;

	//Input
	virtual Input::List		getInputs() const = 0;
	virtual Input::List 	getConnectedInputs() const = 0;
	virtual InputRef  		getInput( const std::string & inLabel ) const = 0;
	virtual FileInputRef	createFileInput( const std::string & inUri ) = 0;
	virtual NetworkInputRef	createNetworkInput( const std::string & inUri ) = 0;
	virtual bool 		 	destroyFileInput( FileInputRef inInput ) = 0;
	virtual bool		 	destroyNetworkInput( NetworkInputRef inInput ) = 0;

	//Output
	virtual Output::List 	getOutputs() const = 0;
	virtual OutputRef		getOutput( const std::string & inLabel ) const = 0;

	//Error
	virtual ErrorCode		getError() const = 0;

	//System
	virtual SystemRef		getSystem() const = 0;

	//Audio
	virtual AudioRef		getAudio() const = 0;

	//Callbacks
	CallbacksRef            getCallbacks() const { return fCallbacks; }
	
	//ExtendedError
	// Returns true if there is an extended error and the out parameters have been populated.
	// Returns false if there is no extended error.
	virtual bool			getExtendedError( int & outErrorCode , int & outSubErrorCode , std::string & outDescription ) const { return false; }



protected:

	void			setCallbacks( CallbacksRef inCallbacks ) { fCallbacks = inCallbacks; }

	virtual ~MasterControl() {}

private:

    MasterControl() {}

    CallbacksRef fCallbacks;
};

//-------------------------------------------------------------
//  CONTROL
//-------------------------------------------------------------

class KON_EXPORTABLE Control : public Retainable
{
public:

	enum ControlType
	{
		TypeNotAvailable = 0,
		TypePlayback,
		TypeTuner
	};

	//Standard Interface
	virtual ControlType	getType() const = 0;

protected:

	virtual ~Control(){};
};

class KON_EXPORTABLE PlaybackControl : public Control
{
public:

	enum State
	{
		Play = 0,
		Paused,
		FF,
		RW,
		Stopped,
		Buffering,
		Empty,
		InfoLoaded,
		Eof,
		Unknown,
		Error
	};

	enum PlaybackControlSupport
	{
		doesPlay = 0,
		doesPlayImage,
		doesStop,
		doesPause,
		doesStep,
		doesSetSpeed,
		doesSetTimeIndex,
		doesNextChapter,
		doesPrevChapter,
		doesBuffer,
		doesResumePlay
	};

	virtual bool		play() = 0;
	virtual bool		stop() = 0;
	virtual bool		pause() = 0;
	virtual bool		step( long inStep ) = 0;

	virtual bool		buffer( long inBufferMsecs ) = 0;

	virtual bool		setSpeed( long inSpeed ) = 0;
	virtual long		getSpeed() const = 0;

	virtual long		getTimeIndex() const = 0;
	virtual bool		setTimeIndex( long inIndex ) = 0;

	virtual long 		getBufferedDuration() const = 0;
	virtual int		getBufferedPercentage() const = 0;
	virtual bool 		getIsBufferFull() const = 0;

	virtual long		getDuration() const = 0;
	virtual State		getState() const = 0;

	virtual bool		nextChapter() = 0;
	virtual bool		prevChapter() = 0;

	virtual bool		doesSupport( PlaybackControlSupport inSupport ) const = 0;

protected:

	virtual ~PlaybackControl() {}
};

class KON_EXPORTABLE TunerControl : public PlaybackControl
{
public:

	enum TunerControlSupport
	{
		doesSetChannel = 0,
		doesUpChannel,
		doesDownChannel,
	};

	virtual ChannelRef	getCurrentChannel() const = 0;
	virtual bool		setChannel( ChannelNum inChannel ) = 0;
	virtual ProgramRef	getCurrentProgram() const = 0;
	virtual bool		upChannel() = 0;
	virtual bool		downChannel() = 0;
	virtual Broadcast	getBroadcastType() const = 0;
	virtual bool		doesSupport( TunerControlSupport inSupport ) const = 0;

protected:

	virtual ~TunerControl() {}
};

}; //End Namespace

#endif //__TVAPI_H__
