/*
 *****************************************************************************
 * Gallery update service
 *****************************************************************************
*/ 

/*
	Gallery error class
	Used to throw exceptions when we know the situation
*/	

function GalleryError()
{
    this.exceptionStatus = null;
    
    // for GalleryError( code , some Error )
    
    if (arguments.length === 2 && (arguments[1] instanceof Error)) 
    {
        this.code = arguments[0];
        this.message = arguments[1].message;
    }
    
    // for GalleryError( code , engine exception )
    
    if (arguments.length === 2 && (arguments[1] instanceof Exception)) 
    {
        this.code = arguments[0];
        this.message = arguments[1].toString();
        this.exceptionStatus = arguments[1].code;
    }
    
    // for GalleryError( code , message )
    
    else if (arguments.length === 2) 
    {
        this.code = arguments[0];
        this.message = arguments[1];
    }
    
    // for GalleryError( another GalleryError )
    
    else if (arguments.length === 1 && (arguments[0] instanceof GalleryError)) 
    {
        this.code = arguments[0].code;
        this.message = arguments[0].message;
        this.exceptionStatus = arguments[0].exceptionStatus;
    }
    
    // for GalleryError( some Error )
    
    else if (arguments.length === 1 && (arguments[0] instanceof Error)) 
    {
        this.code = GalleryErrorCodes.UNKNOWN_ERROR;
        this.message = arguments[0].message;
    }
    
    // for GalleryError( something we can toString )
    
    else if (arguments.length == 1) 
    {
        this.code = GalleryErrorCodes.UNKNOWN_ERROR;
        
        try 
        {
            this.message = arguments[0].toString();
        } 
        catch (e) 
        {
            this.message = "Unable to get error message";
        }
    }
    
    // Everything else
    
    else 
    {
        this.code = GalleryErrorCodes.UNKNOWN_ERROR;
        this.message = "This error object was created with something weird";
    }
}

GalleryError.prototype.toString = function()
{
    // Get the name for the error code
    
    var name = null;
    
    for (var p in GalleryErrorCodes) 
    {
        if (GalleryErrorCodes[p] === this.code) 
        {
            name = p;
            break;
        }
    }
    
    if (!name) 
        name = String(this.code);
    
    // Get the name for the engine exception
    
    var ex = null;
    
    if (this.exceptionStatus) 
    {
        for (var p in Widgets) 
        {
            if (Widgets[p] === this.exceptionStatus) 
            {
                ex = p;
                break;
            }
        }
        
        if (!ex) 
            ex = String(this.exceptionStatus);
    }
    
    return "(" + name + ")" + (ex ? " (" + ex + ")" : "") + " [" + this.message + "]";
}

/*
	Gallery error codes
	These are sent to the onError function of the callback object as the "code"
	property of a GalleryError object.
*/	

var GalleryErrorCodes = 
{
	//.........................................................................	
	// We caught an exception and we don't know what it is
	 
	UNKNOWN_ERROR			: -1,
	
	//.........................................................................	
	// Something went wrong preparing the request. It could
	// be that an engine API call to get data used to prepare
	// the request failed
	
	REQUEST_PREPARE_FAILED	: 10,
	
	//.........................................................................	
	// We were unable to send the request	
	
	REQUEST_SEND_FAILED		: 11,
	
	//.........................................................................	
	// The request was sent but the response never came back.
	// This could be because the network is out, or the 
	// gallery is down, etc. 
	
	RESPONSE_FAILED			: 20,
	
	//.........................................................................	
	// The response came back from the gallery but the HTTP
	// status code is not 200, signaling an error from the
	// gallery.
	
	RESPONSE_IS_ERROR		: 21,
	
	//.........................................................................	
	// The response came back but we could not understand it.
	// It could be that we could not parse the body.
	
	RESPONSE_IS_BAD			: 22,
	
	//.........................................................................
	// Something went bad when we were trying to process the response from the
	// gallery.
	
	FAILED_TO_PROCESS_RESPONSE : 23,	

	//=========================================================================
	// The ones below happen for individual updates ===========================
	//=========================================================================
		
	//.........................................................................	
	// We were unable to create and start a download job. 
	// We could get several of these - any one does not
	// necessarily mean that we should abort the whole process.
	
	DOWNLOAD_CREATE_FAILED	: 30,
	
	//.........................................................................	
	// The download job failed, for whatever reason. 
	// We could get one of these for each download.
	
	DOWNLOAD_FAILED			: 31,
	
	//.........................................................................	
	// Engine API failures. Self explanatory. These could
	// happen several times, one for each update.
	// These will have an engine exception that makes them more
	// granular.
		
	UNINSTALL_FAILED			: 42,
	INSTALL_FAILED				: 43,
	
};

/*
  Utility to simulate errors. It is only turned on if we are NOT
  on a production build and the environment variable 
  KF_BS_SIMULATE_ERROR is set 
 */

var _SIMULATE_ERROR_ = function() {};

if (!_PRODUCTION ) 
{
    var codeSet = getenv("KF_BS_SIMULATE_ERROR");
	
	if (codeSet) 
	{
		log("\n\n\t\t\tGALLERY ERROR SIMULATION IS ON\n\n.");
		
		_SIMULATE_ERROR_ = function(code, msg)
		{
			for (var p in GalleryErrorCodes) 
			{
				if (GalleryErrorCodes[p] == code && p == codeSet ) 
				{
					log("** SIMULATED ERROR ***");
					throw new GalleryError(code, "*SIMULATED* : " + (msg ? String(msg) : ""));
				}
			}
		};
	}
}

// define gallery update service
var GalleryUpdate = 
{
  /** properties **/
  
  // service id
  id: 'Updater',

  // update task id
  task_id: 'service.update.task.0',

  // reference to global task
  task: null,

  // time in seconds since start of day when update should start
  default_update_time: 3 * 60 * 60,  // every day at 3am
  update_time: null,

  // size of update window in seconds, within which the update would run
  default_update_window: 60 * 60,  // one hour
  update_window: null,
  
  // minimum start date buffer in seconds for task
  min_start_interval: 60,
  max_stop_years: 10,
  
  // update uri_part
  gallery_update_uri: 'updates',

  // delayed callbacks waiting for downloads
  session: {},

  // gallery response actions
  CURRENT:  'current',  // can keep it
  UPDATE:   'update',   // should be updated
  UPGRADE:  'upgrade',  // update required
  INSTALL:  'install',  // install it
  DELETE:   'delete',   // recommended delete
  KILL:     'kill',     // must delete now

  // gallery download types
  WIDGET_TYPE_NONE: 0,
  WIDGET_TYPE_TV_ENGINE: 1,
  WIDGET_TYPE_TV_PACKAGE: 2,
  //WIDGET_TYPE_TV_FRAMEWORK: 3,
  WIDGET_TYPE_TV_SYSTEMWIDGET: 4,
  WIDGET_TYPE_TV_YAHOOWIDGET: 5,
  WIDGET_TYPE_TV_USERWIDGET: 6,
  
  /**  methods **/
 
  getTVPlugin : function()
  {
      if (typeof tv == "undefined" || !("system" in tv)) 
      {
          var result = {};
		  
          result.system = 
          {
              'OEM': 'OEM1',
              'deviceBrand': 'Yahoo!',
              'deviceClass': 'DevBuild',
              'deviceModel': 'SDL',
              'deviceId': '1234567890',
              'deviceVersion': 'Pre-Alpha',
              'softwareVersion': 'Alpha',
              'country': 'US',
              'locationCode': '94089',
              'menuLanguage': 'en',
              'deviceInfo': '',
          };
		  
		  return result;
      }
	  else
	  {
	      return tv;	
	  }
  },
 
  //...........................................................................

  getWidgetUpdates: function f_UP_getWidgetUpdates(a_firstRun, a_callback) 
  {
    function galleryPing()
    {
        try
        {
            new URL().fetch( configSetting.getValue( "gallery.gallery-url" ) + "/ping" , 5 );
        }
        catch( e )
        {
            log( "FAILED TO PING GALLERY :" + e.toString() );
        }
    }

    // CANNOT throw    

	try
	{
		try
		{
            _SIMULATE_ERROR_( GalleryErrorCodes.REQUEST_PREPARE_FAILED );

	    	var update_url = this.getUpdateURL(this.gallery_update_uri,a_firstRun);
	    	var signed_url = this.signURL(update_url);
	    }
	    catch( e )
	    {
	    	throw new GalleryError( GalleryErrorCodes.REQUEST_PREPARE_FAILED , e ); 
	    }
	
	    try
	    {
            _SIMULATE_ERROR_( GalleryErrorCodes.REQUEST_SEND_FAILED );
            
            // On a very first run, we ping the gallery synchronously to establish a B cookie
            
            if ( a_firstRun )
                galleryPing();

 	    	this.makeRequest(signed_url, this.getResponseHandler(this, a_firstRun, a_callback ));
	   	}
	   	catch( e )
	   	{
	   		throw new GalleryError( GalleryErrorCodes.REQUEST_SEND_FAILED , e );
	   	}
	}
	catch( e )
	{
		// If anything above failed, we are done
		
		a_callback.onError( new GalleryError( e ) );
		a_callback.onFinished();
	}	   	
  },

  //...........................................................................

  getUpdateURL: function f_UP_getUpdateURL(a_uri, a_firstRun) 
  {
	
	// In order to check if developer mode is enabled, we have to look
	// at each profile's data
	 
 	function developerMode()
 	{
 	    var profiles = profile.getIDs();
 	    
 	    var enabled = null; 	    
 	    var code = null;
 	    
 	    for( i = 0; i < profiles.length; ++i )
 	    {
 	        try
 	        {
 	            var data = persist.getAppData( profiles[ i ] , "com.yahoo.widgets.tv.widgetgallery" );
 	            
 	            if ( code === null )
     	            code = data.get( "gallery.developer-code" );
 	            
 	            var s = data.get( "gallery.show-test-widgets" );
 	            
 	            if ( s === "true" )
 	                enabled = true;
 	            else if ( s == "false" && ! enabled )
 	                enabled = false; 	            
 	        }
 	        catch(e)
 	        {
 	        }
 	    }
 	    
 	    // If we found something in the new area, we return here
 	    
 	    if ( enabled === true )
 	        return code;
 	    
 	    if ( enabled === false )
 	        return null;
	      
	    // If enabled is null, then we didn't find anything above
	    
	    code = null;
 	    
 	    // Otherwise, we look in the old area
 	    
 	    for( i = 0; i < profiles.length; ++i )
 	    {
 	        try
 	        {
 	            var data = persist.getProfileData( profiles[ i ] );
 	            
 	            if ( code === null )
 	                code = data.get( "gallery-developer-code" );
 	            
 	            
 	            if ( data.get( "gallery-developer-showtestwidgets" ) === "true" )
                    enabled = true;
 	        }
 	        catch(e)
 	        {
 	        }
 	    }
 	    
 	    if ( enabled === true )
 	        return code;
 	        
 	    return null;
 	}


  	// CAN throw
	
    var url = configSetting.getValue( "gallery.gallery-url" ) + "/";
    
    url += a_uri;
    url += "?format=json";

    if (a_firstRun)
    {
        url += "&bootstrap=1";
    }
    else
    {
        // When it is not a first run, we send the last time we spoke
        // to the gallery - which we may have stored from its Date
        // header in a previous run. This lets the gallery throttle
        // us at its own discretion.
        
	    var lastTime = currentAppConfig.get( "last-gallery-time" );
	    
	    if ( lastTime )
	    {
            url += "&last=" + lastTime;
        }
    }
    
    var devcode = developerMode();

    if ( devcode )
    {
        url += "&devcode=" + String( devcode );
    }
    
    var devinfo = this.getDevInfo();
    
    for(var key in devinfo) 
	{
      if(devinfo[key]) 
	  {
        url += "&" + key + "=" + escape(devinfo[key]);
      }
    }
        
    var frameworkVersion = null;
    var containerVersion = null;

    var widgetsinfo = this.getWidgetsInfo();

    var widgets_uri = '';
    
    for(var i=0; i<widgetsinfo.length; i++) 
	{
      widgets_uri += ( i ? ',' : '' );
      widgets_uri += String( widgetsinfo[i].id ) + "-" + String( widgetsinfo[i].version );
    
      // This sucks - we have to send the container and framework version separately, 
      // so we have to look for their IDs 
        
      if ( widgetsinfo[i].id == "com.yahoo.widgets.tv.packages.framework" )
      	frameworkVersion = String( widgetsinfo[i].version );
      else if ( widgetsinfo[i].id == "com.yahoo.widgets.tv.container" )
      	containerVersion = String( widgetsinfo[i].version );
    }

    url += '&installed=' + widgets_uri;
    
    url += '&ts=' + String(getRealTime());
    
    if ( frameworkVersion )
    	url += "&fwversion=" + frameworkVersion;
    if ( containerVersion )
    	url += "&dockversion=" + containerVersion;
    
    return url;
  },

  //...........................................................................

  signURL: function f_UP_signURL(a_url) 
  {
  	// CAN throw
	
    return a_url + '&sig=' + md5(a_url + superConfigSettings.getValue("gallery-secret"));
  },

  //...........................................................................

  getDevInfo: function f_UP_getDevInfo() 
  {
    // CAN throw
    
	var tvsys = GalleryUpdate.getTVPlugin().system;
	
    var devinfo = {};
    
    devinfo['man'       ] = tvsys.OEM || null;
    devinfo['brand'     ] = tvsys.deviceBrand || null;
    devinfo['class'     ] = tvsys.deviceClass || null;
    devinfo['model'     ] = tvsys.deviceModel || null;
    devinfo['deviceid'  ] = tvsys.deviceId || null;
    devinfo['hwversion' ] = tvsys.deviceVersion || null;
    devinfo['swversion' ] = tvsys.deviceSoftwareVersion || null;

    devinfo['region'    ] = yahooSettings.get('countryCode') || tvsys.country || null;
    devinfo['location'  ] = yahooSettings.get('locationCode') || tvsys.locationCode || null;

    devinfo['lang'      ] = tvsys.menuLanguage || null;
    devinfo['deviceinfo'] = tvsys.deviceInfo || null;
    devinfo['yweversion'] = konfabulatorVersion() || null;
    devinfo['yweres'    ] = screen.width + 'x' + screen.height;
    
	
    return devinfo;
  },
  
  //...........................................................................

  getWidgetsInfo: function f_UP_getWidgetsInfo() 
  {
    // CAN throw
    
    var widgets_list = [];

    widgets_list = widgets.getUpdateInventory();
    
    var widget_info = [];
    var w;

    for(var i =0; i<widgets_list.length; i++) 
	{
      w = widgets_list[i];
      
      log("HAVE:" + JSON.stringify(w));
      
      var o = {};
      
      o["id"] = w.identifier;
      o["version" ] = w.version;

      widget_info.push( o );
    }
     
    return widget_info;
  },
  
  //...........................................................................

  getResponseHandler: function f_UP_getResponseHandler(a_ctx, a_firstRun, a_callback) 
  {
  	// CAN throw here
	
    var result = 

        function () 
        {
			// CANNOT throw here
			
            if( this.readyState == 4 ) 
            {
                log( "REQUEST DONE : " + String( this.status ) );
                 
                if ( this.status == 204 )
                {
                    // The gallery sends back a 204 when we called it too soon.
                    
                    log( "GALLERY SAYS HOLLABACK" );
					
                    a_callback.onFinished();
                }
                else
                {
                    var oops = (this.status != 200);
        
                    a_ctx.onGotWidgetUpdates(
                        { 
                            err         : oops, 
                            response    : this.responseText, 
                            firstRun    : a_firstRun, 
                            callback    : a_callback , 
                            headers     : this.getAllResponseHeaders()
                        });
                }                         
            }
        };
        
    return result;
  },
  
  //...........................................................................

  makeRequest: function f_CON_makeRequest(a_url, a_callback) 
  {
  	// CAN throw
	
    if(this.hasConnection() == false) 
	{
      throw new Error( 'Internet connection unavailable' );
    }
    
    var request = new XMLHttpRequest();

    request.onreadystatechange = a_callback;

    request.open("GET", a_url, true);

    log('Sending request: ' + a_url);

    request.send();
  },

  //...........................................................................

  hasConnection: function f_UP_hasConnection() 
  {
		return true;
  },

  //...........................................................................

  onGotWidgetUpdates: function f_UP_onGotWidgetUpdates(a_body) 
  {
  	// CANNOT throw
	
    log("GOT GALLERY RESPONSE");
    
    log(a_body.response);

	// First we try to parse the response. Even if it was an error 
	// (HTTP status code != 200) the body may contain an error 
	// description.
	
	var response = null;
	 
	if (a_body.response !== "" ) 
	{
		try
		{
            _SIMULATE_ERROR_( GalleryErrorCodes.RESPONSE_IS_BAD , "Parsing JSON" );

	    	response = JSON.parse(a_body.response);
	    }
	    catch( e )
	    {
	    	// We failed to parse the response, there is nothing more we can do
	    	
	    	a_body.callback.onError( new GalleryError( GalleryErrorCodes.RESPONSE_IS_BAD , e ) );
	    	a_body.callback.onFinished();
	    	return false;
	    }
	}	    

	// The response was an error, we try to get the error information
	
	try
	{
        _SIMULATE_ERROR_( GalleryErrorCodes.RESPONSE_IS_ERROR , "")
	}
	catch(e)
	{
		a_body.err = true;
	}
	
	
	if (a_body.err) 
	{
		var message = "Unknown message";
		
		try
		{
			message = response.error.description + ":" + response.error.detail;	
			
		}
		catch( e )
		{
		}
		
    	a_body.callback.onError( new GalleryError( GalleryErrorCodes.RESPONSE_IS_ERROR , message ) );
    	a_body.callback.onFinished();
    	return false;
	}
	
	// If the response was empty
	
	try 
	{
		_SIMULATE_ERROR_( GalleryErrorCodes.RESPONSE_IS_BAD ); 
	}
	catch( e )
	{
		response = null;
	}
	
	
	if ( response === null ) 
	{
	   	a_body.callback.onError( new GalleryError( GalleryErrorCodes.RESPONSE_IS_BAD , "Empty response" ) );
	   	a_body.callback.onFinished();
	   	return false;	
	}

	// If something goes wrong here, we declare it finished.
	// Note that handleUpdate (which is done in a loop) should not bubble exceptions out here
	// Because those should be related to individual update jobs and not the update process
	// as a whole
	
	try
	{
		_SIMULATE_ERROR_( GalleryErrorCodes.FAILED_TO_PROCESS_RESPONSE );
		
        try
        {
            // OK, the response appears to be good, so we are going to store the date we get from its
            // "Date" header as the last date we got a good reponse. We will pass this back to
            // the gallery in future calls so that it can throttle us.
            
            var dateHeader = null;
            
            for( var i = 0; i < a_body.headers.length; i++ )
            {
                if ( a_body.headers[ i ].split( ":" , 1 ) == "Date" )
                {
                    dateHeader = a_body.headers[ i ].slice( "Date:".length );
                    break;
                }
            }
             
            if ( dateHeader )
            {
                // Convert to seconds
                
                savedDate = String( Math.floor( Date.parse( dateHeader ) / 1000 ) );
                
                currentAppConfig.set( "last-gallery-time" , savedDate );
                
                log( "STORED last-gallery-time AS " + savedDate );
            }            

	        // Nothing to do
	        
	        if(parseInt(response.total) == 0) 
	        {
	          log('nothing to update');
	    
	          a_body.callback.onFinished();
	          return false;
	        }

        }
        catch( e )
        {
            // If something goes wrong, we don't really care too much
        }
        
        // This is the array of updates included in the response
        	
	    var data = response.updates;
	    	    
	    // Nothing to do
	    
	    if( data.length == 0 )   
	    {
	      a_body.callback.onFinished();
	      return false;
	    }
	    	
	    // Delete any old download jobs
	    
	    try 
		{
			this.deleteOldJobs();
		}
		catch(e)
		{
	       // Not deadly		
		}

	    
	    var session_id = (new Date()).getTime() + Math.random();
	
	    var me = this;
	    
	    var session =
	    	{ 
	    		daddy	: me,
	    		id    	: session_id,
	    		left	: response.updates.length,
	    		
	    		callback: a_body.callback, 
	    		total	: response.updates.length, 
	    		finished: 0,
	    		
	    		onUpdateFinished : function( error , widgetID , updateType )
	    			{
	    				log( "FINISHED UPDATE : " + String( this.left-1 ) + " LEFT" );
	    				
	    				if ( error )
	    				{
	    					this.callback.onError( error , widgetID , updateType );
	    				}
	    				
	    				this.left--;
	    				
	    				if ( this.left <= 0 )
	    				{
	    					this.daddy.session[this.id] = null;
	    					delete this.daddy.session[this.id];
	    					
	    					this.callback.onFinished();
	    				}
	    				else
	    				{
	    					this.finished++;
	    					this.callback.onProgress( "" , this.finished , this.total );
	    				}
	    			},
	    
	    		onUpdateFailed : function( error , widgetID , updateType )
	    			{
	    				log( "UPDATE FAILED : " + error.toString() );
	    				
	    				this.onUpdateFinished( error , widgetID , updateType );
	    			}	
	    	};

	    this.session[session_id] = session;
	
		
        // Collect all the widget IDs from the update so we can save them as the 
        // "app order" - this lets the gallery control the order of snippets in
        // the dock.
        
        if ( a_body.firstRun )
        {
            try
            {
                var widget_ids = [];
                
                for( var i = 0; i < data.length; i++) 
                {
                    switch ( data[i].update.action )
                    {
                        case this.UPDATE:
                        case this.UPGRADE:
                        case this.INSTALL:
                        case this.CURRENT:
                        {                
                            switch( parseInt( data[i].update.type ) )
                            {
                                case this.WIDGET_TYPE_TV_USERWIDGET:
                                    widget_ids.push( data[i].update.wid ); 
                                    break;
                                case this.WIDGET_TYPE_TV_SYSTEMWIDGET:
                                case this.WIDGET_TYPE_TV_YAHOOWIDGET: 
    
                                    // This serves to skip widgets that should not be in the dock - 
                                    // such as OOBE, Bootstrap and the dock itself. They are marked
                                    // this way in config.xml, in the group "default-profile-widgets".
                                    // They have the value "NEVER".
                                    
                                    if ( ! widgets.isNeverDefaultProfileWidget( data[i].update.wid ) )                                 
                                        widget_ids.push( data[i].update.wid );

                                    break;
                            }
                        }                    
                    }
                }
    
                if ( widget_ids.length > 0 )
                    yahooSettings.set( "gallery-app-order" , widget_ids.join( "," ) );
                    
            }
            catch( e )
            {
            }
        }         
	
        // Now process the updates	    
	
	    var update;
	    
	    a_body.callback.onProgress( "" , session.finished , session.total );	    	    
	    
	    for( var i = 0; i < data.length; i++) 
	    {
	       update = data[i].update;
	
	       // This should not bubble any exceptions out here - because those
	       // are related to individual updates and not the process as a whole.
	       
	       this.handleUpdate( update , session_id , a_body.firstRun );
	    }	
	}
	catch( e )
	{
		a_body.callback.onError( new GalleryError( GalleryErrorCodes.FAILED_TO_PROCESS_RESPONSE , e ) );
		a_body.callback.onFinished();
	}
  },

  deleteOldJobs: function f_UP_removeOldJobs() 
  {
  	// CAN throw
	
    var jobs = download.getJobs();
    var job;
    
    for(var i=0; i<jobs.length; i++) 
	{
      job = jobs[i];
      job.removeLocalFile();

      log('removing old job: ' + job.uri);

      download.deleteJob(job.id);
    }
  },

  handleUpdate: function f_UP_handleUpdate(a_update, a_session_id, a_firstRun) 
  {
  	// CANNOT throw
	
    log( " " );
    log( "HANDLING UPDATE '" + a_update.action + "' : '" + a_update.id + "'" );

    var session = this.session[a_session_id];

	try
	{
	    var wid = a_update.wid;
	    		    
	    if(wid === "") 
	    {
	    	// A goofy update
	    	session.onUpdateFinished();
	    	return null;
	    }
	    
	    
	    // based on update action
	    switch(a_update.action) 
	    {
	      //...................................................................
	      // Update, upgrade and install require us to go and download stuff
	    	
	      case this.UPDATE:
	        // upgrade widget
	        a_update.widgetfile && this.downloadAndInstall(a_update, a_session_id, false);
	
	        break;
	
	      case this.UPGRADE:
	        // upgrade widget
	        a_update.widgetfile && this.downloadAndInstall(a_update, a_session_id, (a_firstRun)? false : true );
	
	        break;
	
	      case this.INSTALL:
	        // install widget
	        a_update.widgetfile && this.downloadAndInstall(a_update, a_session_id, (a_firstRun)? false : true);
	        
	        break;

          //...................................................................
	      // This one is a no-op
	        
	      case this.CURRENT:
	    	// Nothing to do
	    	session.onUpdateFinished();
	    	
	    	break;
	    
	      //...................................................................
	      // 
	    	
	      case this.DELETE:
	      case this.KILL:	
	        try 
	        {
               _SIMULATE_ERROR_( GalleryErrorCodes.UNINSTALL_FAILED );

               widgets.uninstallWidgetLater(wid);
	        }
	        catch(e) 
	        {
	           throw new GalleryError( GalleryErrorCodes.UNINSTALL_FAILED , e );
	        }
	
	        // If no exceptions, this update is done
	        
	        session.onUpdateFinished();
	        
	        break;
	        
	      //...................................................................
	      // I have no earthly idea what this is
	        
	      default:
	        log( "GOT UNRECOGNIZED UPDATE ACTION '" + a_update.action + "' FROM GALLERY (IGNORING)" );
	        session.onUpdateFinished();
	        break;
	    }
    	return wid;
    }
    catch( e )
    {
    	session.onUpdateFailed( new GalleryError( e ) , wid );	
    }
  },

  downloadAndInstall: function f_UP_downloadAndInstall(a_update, a_session_id, a_installnow) 
  {
  	// CAN throw
	
    this.downloadFile(a_update, a_session_id, a_installnow);
  },
  
  downloadFile: function f_UP_downloadFile(a_update, a_session_id, a_installnow) 
  {
  	// CAN throw
	
    try
    {
		_SIMULATE_ERROR_( GalleryErrorCodes.DOWNLOAD_CREATE_FAILED );
		
	    log('downloading file: ' + a_update.widgetfile);
	
	    // create job
	    var job = this.createDownloadJob(a_update);
	
	    // assign status handler
	    job.onStatusChanged = this.getStatusChangedHandler(this, a_update, a_session_id, a_installnow);
	  
	    // run job
	    log('running job: ' + job.uri);
	    
	    var session = this.session[a_session_id];
	    
	    // The job needs to be bolted down so it doesn't get lost
	    // in a GC
	    
	    if ( ! session.jobs )
	       session.jobs = [];
	       
	    session.jobs.push( job );
	    	
	    job.start();	    	
	}
	catch( e )
	{
		throw new GalleryError( GalleryErrorCodes.DOWNLOAD_CREATE_FAILED , e );	
	}	    
  },

  createDownloadJob: function f_UP_createDownloadJob(a_update) 
  {
    // CAN throw
    
    var uri = a_update.widgetfile + "?ts=" + String(getRealTime());
    
    var devinfo = this.getDevInfo();
    
    for(var key in devinfo) 
	{
      if(devinfo[key]) 
	  {
        uri += "&" + key + "=" + escape(devinfo[key]);
      }
    }
    
    uri = this.signURL( uri );
    
    // TODO: how should this be set
    var rating = new Rating('MPAA', 'G');
    var title = a_update.wid;
    var approxSize = 0;
    var timeout = 60;
    var extension = a_update.id + '-' + a_update.action + '.widget';
    var access = Job.APP;
    var metadata = { update: JSON.stringify(a_update) };
    
    var job = download.createJob(
      uri,
      rating,
      title,
      approxSize,
      timeout,
      extension,
      access,
      metadata
    );

    return job;
  },

  getStatusChangedHandler: function f_UP_getStatusChangedHandler(a_ctx, a_update, a_session_id, a_install_now) 
  {
  	// CAN throw here
	
    return (
    
    function onStatusChangedHandler() 
    {
        // CANNOT throw here - this is an engine callback
                 
        // Get the session
        
        var session = a_ctx.session[a_session_id];
      
        if ( ! session )
    	   return;

        var status = this.status;
		
		try
		{
			_SIMULATE_ERROR_( GalleryErrorCodes.DOWNLOAD_FAILED );
		}		   
		catch(e)
		{
			status = Job.ERROR;
		}
    
        switch (status)
        {
            case Job.ERROR:
                
                session.onUpdateFailed(new GalleryError(GalleryErrorCodes.DOWNLOAD_FAILED, String(this.errorCode) + " : " + this.error), a_update.wid);
                break;
                
            case Job.COMPLETED:
                
                try 
				{
                    try 
                    {
                        if (a_update.type == GalleryUpdate.WIDGET_TYPE_TV_PACKAGE) 
                        {
                            _SIMULATE_ERROR_(GalleryErrorCodes.INSTALL_FAILED);
                            
                            log('assign package for later install: ' + this.localURI);
                            
                            widgets.installPackageLater(this.localURI, a_update.hash);
                            
                            log('done assigning package for later install');
                        }
                        else 
                        {
                            _SIMULATE_ERROR_(GalleryErrorCodes.INSTALL_FAILED);
                            
                            log('assign widget for later install: ' + this.localURI);
                            
                            var wid = widgets.installWidgetLater(this.localURI, a_update.hash, parseInt(a_update.type), a_update.action);
                            
                            log('done assigning widget for later install: ' + wid);
                        }
                    } 
                    catch (e) 
                    {
                        throw new GalleryError(GalleryErrorCodes.INSTALL_FAILED, e);
                    }
                    
                    session.onUpdateFinished();
                } 
                catch (e) 
				{
                    session.onUpdateFailed(new GalleryError(e), a_update.wid , parseInt( a_update.type ) );
                }
                
                break;
                
            default:
                
                // Don't really care about other job statuses
                
                break;
        }
    });
  }
};

